/**
 * \file: level_db.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * \component: authorization level daemon
 *
 * \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
 *
 * \copyright (c) 2010, 2011 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <dirent.h>
#include <limits.h>
#include <sys/stat.h>
#include <errno.h>

#include "model/level_configuration.h"

#include "create_signature_db_cmd/level_db.h"
#include "create_signature_db_cmd/feature_db.h"
#include "create_signature_db_cmd/signature_db_writer.h"
#include "encryption/signature.h"


static void level_db_check_level_scripts(level_configuration_t *level_conf, const char *root_dir, const char *sysroot_dir);
static error_code_t level_db_check_level_conf(security_level_t level, const char *root_dir, const char *sysroot_dir);
static void level_db_check_level_script(const char *path, const char *message, security_level_t level,
		change_type_t type, bool is_permanent);

static error_code_t level_db_sign_level_scripts(security_level_t level, const char *root_dir);
static error_code_t level_db_sign_level_script(const char *path);

error_code_t level_db_check_level_consistency(const char *root_dir, const char *sysroot_dir)
{
	error_code_t result=RESULT_OK;
	DIR *dp;
	struct dirent *ep;
	char level_scripts_path[PATH_MAX];

	//used to ensure that the maximal level has a public key inside
	//(maximal level private key can be used to change to all other levels as well)
	security_level_t level_max=0;
	bool level_max_has_key=false;
	snprintf(level_scripts_path,PATH_MAX,"%s%s%s/",sysroot_dir,root_dir,LEVEL_SUBDIR);
	dp = opendir (level_scripts_path);
	if (dp == NULL)
	{
		printf("Unable to open the level configuration in \'%s%s\'- %s.\n",sysroot_dir,root_dir,strerror(errno));
		return RESULT_INVALID;
	}

	while ((ep = readdir (dp))!=NULL && result==RESULT_OK)
	{
		security_level_t level;
		char *parse_result;
		bool has_pub_key;

		if (strcmp(ep->d_name,"..")==0 || strcmp(ep->d_name,".")==0) continue;

		if (ep->d_type != DT_DIR)
		{
			printf("WARNING: Unexpected element found in feature conf directory: \'%s\'. Ignoring it.\n",ep->d_name);
			continue;
		}

		level=(security_level_t)strtol(ep->d_name,&parse_result,10);
		if (ep->d_name == parse_result || parse_result[0]!='\0')
		{
			printf("WARNING: Invalid level sub directory: %s. The name of a level sub directory"""
					" needs to be a positive number only. Ignoring the sub directory.\n",ep->d_name);
			continue;
		}

		printf("Security Level \'%d\' found.\n\tScanning scripts of this level ...\n",(int)level);
		has_pub_key=level_configuration_check_pubkey(level_scripts_path,level);
		if (has_pub_key)
			printf("\tLevel configuration contains a public key.\n");
		else
			printf("\tLevel configuration does not contain a public key.\n");

		if (level > level_max)
		{
			level_max=level;
			level_max_has_key=has_pub_key;
		}

		printf("\tChecking level configuration for consistency ...\n");
		result=level_db_check_level_conf(level, root_dir, sysroot_dir);
	}

	if (result==RESULT_OK && !level_max_has_key)
	{
		printf("\nERROR: No public key found in maximum level (level number: %d).\n"
				"At least the maximum level needs a public key in its configuration.\n\n", (int)level_max);
		result=RESULT_INVALID;
	}
	(void) closedir (dp);

	return result;
}

error_code_t level_db_fill_signature_db(const char *root_dir, const char *sysroot_dir)
{
	error_code_t result=RESULT_OK;
	DIR *dp;
	struct dirent *ep;
	char level_scripts_path[PATH_MAX];

	snprintf(level_scripts_path,PATH_MAX,"%s%s%s/",sysroot_dir, root_dir,LEVEL_SUBDIR);
	dp = opendir (level_scripts_path);
	if (dp == NULL)
	{
		printf("Unable to open the level configuration in \'%s%s\'- %s.\n",sysroot_dir, root_dir,strerror(errno));
		return RESULT_INVALID;
	}

	while ((ep = readdir (dp))!=NULL && result==RESULT_OK)
	{
		security_level_t level;
		char *parse_result;

		if (strcmp(ep->d_name,"..")==0 || strcmp(ep->d_name,".")==0) continue;

		if (ep->d_type != DT_DIR)
		{
			printf("WARNING: Unexpected element found in feature conf directory: \'%s\'. Ignoring it.\n",ep->d_name);
			continue;
		}

		level=(security_level_t)strtol(ep->d_name,&parse_result,10);
		if (ep->d_name == parse_result || parse_result[0]!='\0')
		{
			printf("WARNING: Invalid level sub directory: %s. The name of a level sub directory"""
					" needs to be a positive number only. Ignoring the sub directory.\n",ep->d_name);
			continue;
		}

		printf("Signing scripts of level \'%d\' ...\n",(int)level);

		result=level_db_sign_level_scripts(level, root_dir);
	}

	(void) closedir (dp);

	return result;
}

static error_code_t level_db_check_level_conf(security_level_t level, const char *root_dir, const char *sysroot_dir)
{
	level_configuration_t *level_conf=NULL;
	char abs_root_path[PATH_MAX];
	error_code_t result;

	snprintf(abs_root_path,PATH_MAX,"%s%s",sysroot_dir,root_dir);
	result=level_configuration_load_level(&level_conf,abs_root_path,level);
	if (result==RESULT_OK)
	{
		//just some warnings are printed out in this function
		level_db_check_level_scripts(level_conf, root_dir, sysroot_dir);

		if (level_configuration_is_permanent_level(level_conf))
			printf("\tThe level configuration defines the level as permanent.\n");
		else
			printf("\tThe level configuration defines the level as transient.\n");

		result=feature_db_check_level_for_feature_completeness(level_conf);
	}

	if (level_conf!=NULL)
		level_configuration_free(level_conf);

	return result;
}

static void level_db_check_level_scripts(level_configuration_t *level_conf, const char *root_dir, const char *sysroot_dir)
{
	char path[PATH_MAX];
	security_level_t level;
	bool is_permanent;

	level=level_configuration_get_security_level(level_conf);
	is_permanent=level_configuration_is_permanent_level(level_conf);

	//check prepare level change script
	snprintf(path,PATH_MAX,"%s%s%s/%d/%s",sysroot_dir,root_dir,LEVEL_SUBDIR,level,PREPARE_LEVEL_CHANGE_FN);
	level_db_check_level_script(path,"prepare a level change",level, ALD_NORMAL_CHANGE, is_permanent);
	//check finalize level change script
	snprintf(path,PATH_MAX,"%s%s%s/%d/%s",sysroot_dir,root_dir,LEVEL_SUBDIR,level,FINALIZE_LEVEL_CHANGE_FN);
	level_db_check_level_script(path,"finalize a level change",level, ALD_NORMAL_CHANGE, is_permanent);
	//check prepare level change recovery script
	snprintf(path,PATH_MAX,"%s%s%s/%d/%s",sysroot_dir,root_dir,LEVEL_SUBDIR,level,PREPARE_LEVEL_CHANGE_RECO_FN);
	level_db_check_level_script(path,"prepare a level change recovery",level, ALD_RECOVERY_CHANGE, is_permanent);
	//check finalize level change recovery script
	snprintf(path,PATH_MAX,"%s%s%s/%d/%s",sysroot_dir,root_dir,LEVEL_SUBDIR,level,FINALIZE_LEVEL_CHANGE_RECO_FN);
	level_db_check_level_script(path,"finalize a level change recovery",level, ALD_RECOVERY_CHANGE, is_permanent);
}

static void level_db_check_level_script(const char *path, const char *message, security_level_t level,change_type_t type,
		bool is_permanent)
{
	struct stat stat_result;
	if (stat(path,&stat_result)==-1)
	{
		//file not there, which is fine in case of a transient level and no recovery scripts
		if (is_permanent || type==ALD_NORMAL_CHANGE)
			printf("WARNING: No script found in level %d to %s- %s.\n", (int)level,message,strerror(errno));
	}
	else
	{
		//file there, which does not make no sense in case of recovery scripts and a transient level
		if (!is_permanent && type==ALD_RECOVERY_CHANGE)
			printf("WARNING: The scripts to %s will never be executed for transient levels.\n", message);
	}
}

static error_code_t level_db_sign_level_scripts(security_level_t level, const char *root_dir)
{
	error_code_t result;
	char path[PATH_MAX];

	//sign prepare level change script
	snprintf(path,PATH_MAX,"%s%s/%d/%s",root_dir,LEVEL_SUBDIR,level,PREPARE_LEVEL_CHANGE_FN);
	result=level_db_sign_level_script(path);

	if (result==RESULT_OK)
	{
		//sign finalize level change script
		snprintf(path,PATH_MAX,"%s%s/%d/%s",root_dir,LEVEL_SUBDIR,level,FINALIZE_LEVEL_CHANGE_FN);
		result=level_db_sign_level_script(path);
	}

	if (result==RESULT_OK)
	{
		//sign prepare level change recovery script
		snprintf(path,PATH_MAX,"%s%s/%d/%s",root_dir,LEVEL_SUBDIR,level,PREPARE_LEVEL_CHANGE_RECO_FN);
		result=level_db_sign_level_script(path);
	}

	if (result==RESULT_OK)
	{
		//sign finalize level change recovery script
		snprintf(path,PATH_MAX,"%s%s/%d/%s",root_dir,LEVEL_SUBDIR,level,FINALIZE_LEVEL_CHANGE_RECO_FN);
		result=level_db_sign_level_script(path);
	}

	if (result==RESULT_OK)
	{
		//sign level conf file
		snprintf(path,PATH_MAX,"%s%s/%d/%s",root_dir,LEVEL_SUBDIR,level,LEVEL_CONF_FN);
		result=level_db_sign_level_script(path);
	}

	if (result==RESULT_OK)
	{
		//sign level public key
		snprintf(path,PATH_MAX,"%s%s/%d/%s",root_dir,LEVEL_SUBDIR,level,LEVEL_KEY_NAME PUBLIC_KEY_EXT);
		result=level_db_sign_level_script(path);
	}

	return result;
}

static error_code_t level_db_sign_level_script(const char *path)
{
	error_code_t result;
	result=signature_db_writer_add_from_file(path);

	//scripts might miss, we checked before that all files needed are present
	if (result==RESULT_FILE_NOT_FOUND)
		return RESULT_OK;
	printf("\tSignature created for \'%s\'\n",path);

	return result;
}
